<?php
declare(strict_types=1);

/**
 * Cron job to refresh sponsor channel stats and auto-close finished campaigns.
 * Run via CLI: php cron_refresh.php
 */

require __DIR__ . '/helpers.php';
require __DIR__ . '/database.php';
require __DIR__ . '/telegram_api.php';

$config = require __DIR__ . '/config.php';

if (php_sapi_name() !== 'cli') {
    echo "This script must be run from CLI.\n";
    exit(1);
}

$logFile = $config['app']['log_file'];

try {
    $pdo = Database::pdo($config['database']);
} catch (Throwable $exception) {
    bot_log($logFile, 'cron_refresh_db_failed', ['error' => $exception->getMessage()]);
    exit(1);
}

bot_log($logFile, 'cron_refresh_start');

$stmt = $pdo->prepare('SELECT * FROM force_channels WHERE is_active = 1 ORDER BY created_at ASC');
if (!$stmt->execute()) {
    bot_log($logFile, 'cron_refresh_query_failed', ['error' => implode(',', $stmt->errorInfo())]);
    exit(1);
}

$channels = $stmt->fetchAll();
if ($channels === false || $channels === []) {
    bot_log($logFile, 'cron_refresh_no_channels');
    exit(0);
}

foreach ($channels as $channel) {
    $channelId = (int) ($channel['id'] ?? 0);
    if ($channelId === 0) {
        continue;
    }

    try {
        $channel = synchronize_force_channel_counters($pdo, $config, $channel);
    } catch (Throwable $exception) {
        bot_log($logFile, 'cron_refresh_sync_failed', [
            'channel_id' => $channelId,
            'error' => $exception->getMessage(),
        ]);
        continue;
    }

    try {
        $channel = finalize_force_channel_if_completed($pdo, $config, $channel);
        if ($channel === null) {
            bot_log($logFile, 'cron_refresh_campaign_closed', ['channel_id' => $channelId]);
        }
    } catch (Throwable $exception) {
        bot_log($logFile, 'cron_refresh_finalize_failed', [
            'channel_id' => $channelId,
            'error' => $exception->getMessage(),
        ]);
    }

    usleep(250000); // sleep 0.25s to avoid hitting API limits
}

bot_log($logFile, 'cron_refresh_end');
