<?php
declare(strict_types=1);

/**
 * توابع تعامل با Bot API تلگرام.
 */

/**
 * اجرای درخواست HTTP به تلگرام.
 * برای بعضی متدها (مثل answerCallbackQuery) مقدار true برمی‌گردد.
 */
function telegram_request(string $token, string $method, array $params = [], array $files = [])
{
    $url = "https://api.telegram.org/bot{$token}/{$method}";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

    if ($files !== []) {
        foreach ($files as $field => $path) {
            $params[$field] = new CURLFile($path);
        }
    }

    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 15);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);

    $response = curl_exec($ch);

    if ($response === false) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new RuntimeException('CURL failed: ' . $error);
    }

    curl_close($ch);

    $data = json_decode($response, true, 512, JSON_THROW_ON_ERROR);
    if (!($data['ok'] ?? false)) {
        throw new RuntimeException('Telegram API error: ' . ($data['description'] ?? 'unknown'));
    }

    return $data['result'] ?? true;
}

/**
 * ارسال پیام متنی همراه با کیبورد اینلاین.
 */
function telegram_send_message(string $token, int $chatId, string $text, array $extra = []): array
{
    return telegram_request($token, 'sendMessage', array_merge([
        'chat_id' => $chatId,
        'text' => $text,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true,
    ], $extra));
}

/**
 * ارسال اکشن تایپینگ برای حس زنده بودن ربات.
 */
function telegram_send_action(string $token, int $chatId, string $action): void
{
    telegram_request($token, 'sendChatAction', [
        'chat_id' => $chatId,
        'action' => $action,
    ]);
}

/**
 * ارسال گروهی فایل برای آپلودهای چندگانه.
 */
function telegram_send_media_group(string $token, int $chatId, array $media): array
{
    return telegram_request($token, 'sendMediaGroup', [
        'chat_id' => $chatId,
        'media' => json_encode($media, JSON_UNESCAPED_UNICODE),
    ]);
}


